import numpy
import pandas
import matplotlib.pyplot as plt

import json
import sklearn.preprocessing
import sklearn.metrics
import torch
import torch.nn as nn
import torch.utils.data
import torch.autograd

def csv_reader(filename):
    tr = pandas.read_csv(filename)
    tr = tr.sort_values("Date")
    tr.set_index("Date", inplace=True)

    cols = tr.columns.tolist()
    if "OpenInt" in cols:
        cols.remove("OpenInt")
    cols.remove("Close")
    cols.append("Close")

    tr = tr[cols]
    return tr


def create_data(filename, scaler, sscaler):
    tr = csv_reader(filename)
    X_train = tr.iloc[:, :-1]
    Y_train = tr.iloc[:, 3:4]

    X_train = scaler.fit_transform(X_train)
    Y_train = sscaler.fit_transform(Y_train)

    X_train_tensor = torch.autograd.Variable(torch.Tensor(X_train))
    Y_train_tensor = torch.autograd.Variable(torch.Tensor(Y_train))

    X_train_tensor_final = torch.reshape(X_train_tensor,
                                         (X_train_tensor.shape[0], 1, X_train_tensor.shape[1]))

    print("Training Shape", X_train_tensor_final.shape, Y_train_tensor.shape)
    #print(X_train_tensor_final)
    #print(Y_train_tensor)

    return X_train_tensor_final, Y_train_tensor

class LSTMnet(nn.Module):
    def __init__(self, num_classes, input_size, hidden_size, num_layers, seq_length):
        super().__init__()
        self.num_classes = num_classes
        self.num_layers = num_layers
        self.input_size = input_size
        self.hidden_size = hidden_size
        self.seq_length = seq_length

        self.lstm = nn.LSTM(input_size=input_size,
                            hidden_size=hidden_size,
                            num_layers=num_layers,
                            batch_first=True)
        self.fc_first = nn.Linear(hidden_size, 64) #fully connected 1
        self.fc       = nn.Linear(64, num_classes) #fully connected last layer

        self.relu = nn.ReLU()

    def forward(self, x):
        h_0 = torch.autograd.Variable(torch.zeros(self.num_layers, x.size(0), self.hidden_size))
        c_0 = torch.autograd.Variable(torch.zeros(self.num_layers, x.size(0), self.hidden_size))

        output, (hn, cn) = self.lstm(x, (h_0, c_0))
        hn = hn.view(-1, self.hidden_size)
        out = self.relu(hn)
        out = self.fc_first(out)
        out = self.relu(out)
        out = self.fc(out)
        return out

def getloss(model, df, criterion, sscaler):
    Xtransform = scaler.transform(df.iloc[:, :-1])
    df_y_mm = sscaler.transform(df.iloc[:, -1:])

    Xtransform = torch.autograd.Variable(torch.Tensor(Xtransform)) #converting to Tensors
    df_y_mm = torch.autograd.Variable(torch.Tensor(df_y_mm))

    Xtransform = torch.reshape(Xtransform, (Xtransform.shape[0], 1, Xtransform.shape[1]))

    with torch.no_grad():
        model.eval()
        train_predict = model.forward(Xtransform)
        loss = criterion(train_predict, df_y_mm)

        data_predict = train_predict.data.numpy()
        dataY_plot = df_y_mm.data.numpy()
        #data_predict = sscaler.inverse_transform(data_predict)
        #dataY_plot = sscaler.inverse_transform(dataY_plot)

        return loss.item(), sklearn.metrics.mean_squared_error(data_predict, dataY_plot)


def runNN(trainX, trainY, valid, tests, scaler, sscaler):
    num_epochs = 3000
    learning_rate = 0.001

    feature_size = 4
    hidden_size = 2
    num_layers = 1
    num_classes = 1

    model = LSTMnet(num_classes, feature_size, hidden_size, num_layers, trainX.shape[1])
    criterion = torch.nn.MSELoss()
    optimizer = torch.optim.Adam(model.parameters(), lr=learning_rate)

    v = []
    for epoch in range(num_epochs):
        model.train()
        optimizer.zero_grad()
        outputs = model.forward(trainX)

        loss = criterion(outputs, trainY)
        loss.backward()
        optimizer.step()

        if epoch % 10 == 0:
            print("Epoch: %d, loss: %1.5f" % (epoch, loss.item()))

            validloss, validmse = getloss(model, valid, criterion, sscaler)
            testloss,  testmse  = getloss(model, tests, criterion, sscaler)

            v.append([epoch, float(validloss), float(validmse), float(testloss), float(testmse)])
            #torch.save(model.state_dict(), "model_epoch-{}_vali-{:.2f}-{:.2f}_test-{:.2f}-{:.2f}.ckpt".format(epoch, validloss, validmse, testloss, testmse))
    with open("ans-hide{}.json".format(hidden_size), "w") as fp:
        json.dump(v, fp, indent = 4)

scaler = sklearn.preprocessing.MinMaxScaler()
sscaler = sklearn.preprocessing.StandardScaler()

trainX, trainY = create_data("train.csv", scaler, sscaler)
valid = csv_reader("validation.csv")
tests = csv_reader("test.csv")

runNN(trainX, trainY, valid, tests, scaler, sscaler)
